﻿<#
    .SYNOPSIS
      Converts and imports a Men and Mice Suite SqLite database to MS SQl Server or PSQL database.
	  This script uses:
	  	System.Data.SQLite dynamically linked library
	  and 
	   C:\Program Files\PostgreSQL\12\bin\psql.exe
	  The default path can be overwritten with parameter psqlCMDpath
	  Default port for PSQL is 5432 and can be set with the param tcpPort
	.INPUTS
	  Input SQLite file and target database information
	.EXAMPLE
     ConvertDatabase.ps1 -sourceDbFile .\mmsuite.db -database mmsuite -ServerInstance localhost
	.EXAMPLE
	 .\ConvertDatabase2.ps1 -sourceDbFile .\mmsuite.db -database mmsuite -DBType PSQL -username postgres -ServerInstance localhost
#>

##############################################################################
## Copyright (c) 2025 BlueCat Networks
##
## ConvertDatabase2.ps1
## Version 3.1
##
## Converts and imports a Men and Mice Suite SqLite database to MS SQL Server database or PSQL DB.
##
## Prerequisites:
## 		An empty, but initiated Men and Mice MS SQL Server or PSQL Database instance must have been
## 		created before this script can be run.
##      For PostgreSQL pls see create script in the psqlscripts subdirectory.
##
## This script uses:
##		System.Data.SQLite from http://system.data.sqlite.org/
##
##############################################################################
param(
	[Parameter(Mandatory=$true)] $sourceDbFile,
	$database = "mmsuite",
	$ServerInstance = "(local)",
	$username=$null,
    $DBType ="MSSQL",
	[switch]$useWindowsAuthentication = $false,
    $tcpPort = $null,
	$batchSize = 500000,
	$dbSchema = 'mmCentral',
	$psqlCMDpath = 'C:\Program Files\PostgreSQL\12\bin\psql.exe',
	[switch]$skipForeignKeyCheck,
	[switch]$ignoreDBVersion = $false,
	[switch]$skipPurgingTables,
	[switch]$forceSharedMemory = $false,
    [switch]$forceNamedPipes = $false,
    [switch]$forceTCP = $false
)

function mmLogger ($msg,$contextInfo,$theLogFile,[switch]$logToConsole,[switch]$asError) {
    function logToFile ($itsRetry) {
        if (-not (isEmpty $msg)) {
            ("{0} [{1}]: {2}" -f $timeStr,$contextInfo,$msg) | out-file -append $theLogFile
        } else {
            " " | out-file -append $theLogFile
        }
    }

    if ($timeStr -eq $null){
        $timeStr = (getCurrentDate)
    }

    if ($logToConsole) {
    	if ($asError){
        	write-host $msg -foregroundcolor Red
    	} else {
    		write-host $msg
    	}
    } elseif ($PSCmdlet.MyInvocation.BoundParameters["Verbose"].IsPresent){
    	write-verbose $msg
    }

    $retry = 0
    while ($true) {
        try {
            logToFile $retry
            break
        } catch {
            start-sleep -m 25
            $retry++
            if ($retry -gt 200){
                break
            }
        }
    }
}

function getCurrentDate () {
    return (get-date -Format 'yyyy-MM-dd HH:mm:ss')
}

function isEmpty([string] $string){
    if ($string -ne $null) { $string = $string.Trim() }
    return [string]::IsNullOrEmpty($string)
}


function loggit ($msg,[switch]$asError,[switch]$logToConsole) {
	mmLogger -msg $msg -contextInfo (Get-PSCallStack)[1].Command -theLogFile $logfile -logToConsole:$logToConsole -asError:$asError
}


function InvokeNonQuery ($query){
	$cn = getSQLiteConnection
	$cm =  getSQLiteCommand $query $cn
	$rowsAffected= $cm.ExecuteNonQuery()
	#$cn.close()
	return $rowsAffected
}



function CheckDatabaseForMigration (){
	# if there are columns that have different size in SQL Server then a special case needs to be added here for those
	# Specialcase example : @("TableName", "columnName", "LengthISQLServer")
	$specialCases = @(	@("mm_customfieldvalues", "customfieldvalue", "892"), 
						@("mm_reportviews", "createstring", "8000")
					)

	$redArgs = @{}
	$redArgs["foregroundColor"] = "Red"

	function InvokeScalar ($query){
		$cn = getSQLiteConnection
		$cm =  getSQLiteCommand $query $cn

		$scalar= $cm.ExecuteScalar()
		#$cn.close()
		
		return $scalar
	}

	function InvokeAndReturnRows ($query){
		$cn = getSQLiteConnection
		$cm =  getSQLiteCommand $query $cn
		$da = new-object System.Data.SQLite.SQLiteDataAdapter($cm)
		[System.Data.DataTable]$dt = new-object System.Data.DataTable("sometable")
		[void]$da.Fill($dt)
		#$cn.close()
		
		return $dt.Rows
	}

	function truncColumn($table, $column, $maxSize)
	{
		$redArgs = @{}
		$redArgs["foregroundColor"] = "Red"
		$command = "update "+$table+" set "+$column+" = substr($column,0, $maxSize - 1) where length("+$column+") > $maxSize ;"
		Write-Host ""
		Write-Host "truncating column $column in table $table sql: `"$command`"" @redArgs
		Write-Host ""
		InvokeNonQuery $command
	}

	function truncColumnWithCheck($table, $column, $maxSize)
	{
		$extraArgs = @{}
	  	$extraArgs["foregroundColor"] = "Yellow"
		Write-Host ""
		Write-Host "Would you like to truncate the values that are to long in table: $table, column: $column."  @extraArgs
		Write-Host "It is recomended that you have a backup of the database before doing this operation. as this "  @extraArgs
		Write-Host "operation can not be undone!"  @extraArgs
		$resp = Read-Host "[Y]es or [N]o" 
		Write-Host ""
		if($resp -eq "y" -or $resp -eq "yes") {	
			truncColumn $table $column $maxSize 
			return 1
		} else {
			Write-Host "The column will not be truncated. Migrating to SQL Server can not be done while "
			Write-Host "there are to long values in columns. Contact Men and Mice support for assistance."
			return 0
		}
	}

	function clearRows($table, $column, $maxSize)
	{
	    $redArgs = @{}
	    $redArgs["foregroundColor"] = "Red"
	    $command = "delete from $table where length($column)>$maxSize;" 
	    Write-Host ""
	    Write-Host "clearing rows in table $table that have too long values in column $column. sql: `"$command`"" @redArgs
	    Write-Host ""
	    InvokeNonQuery $command
	}

	function clearRowsWithCheck ($table,$column,$maxSize){
	    $extraArgs = @{}
	    $extraArgs["foregroundColor"] = "Yellow"
	    Write-Host ""
	    Write-Host "Would you like to clear rows that have values that are to long in table: $table, column: $column."  @extraArgs
	    Write-Host "It is recomended that you have a backup of the database before doing this operation. as this "  @extraArgs
	    Write-Host "operation can not be undone!"  @extraArgs
	    $resp = Read-Host "[Y]es or [N]o" 
	    Write-Host ""
	    if($resp -eq "y" -or $resp -eq "yes") { 
	        clearRows $table $column $maxSize 
	        return 1
	    } else {
	        Write-Host "The column will not be truncated. Migrating to SQL Server can not be done while "
	        Write-Host "there are to long values in columns. Contact Men and Mice support for assistance."
	        return 0
	    }
	}

	function GetAddressAsExpandedString([IPAddress]$inAddress)
	{
		if ($inAddress -eq $null)
		{
			return [string]::Empty
		}
		$bytes = $inAddress.GetAddressBytes()

		$address = [string]::Empty
		foreach ($byte in $bytes)
		{
			$address += "{0:x2}" -f $byte
		}
		return $address
	}

	function GetIPv6Address([string]$inDotted)
	{
		$ipAddress = $null
		if (-not [System.Net.IPAddress]::TryParse($inDotted,[ref]$ipAddress))
		{
			return $null
		}
		if ($inDotted.Contains(".") -and -not $inDotted.Contains(":"))
		{
			$ipAddress = [System.Net.IPAddress]::Parse([string]::Format("::FFFF:{0}", $inDotted))
		}
		return $ipAddress
	}

	function AddToIPAddress([string]$inAddress)
	{
		$theAddress = GetAddressAsExpandedString ( GetIPv6Address ($inAddress))
		if ( -not [string]::IsNullOrEmpty($theAddress))
		{
			$theInsertCommand = [string]::Format("insert into mm_ipaddress (address, dotted, state) values ({0}, '{1}', 0);", $theAddress, $inAddress);
			$theMissingAddresses = InvokeNonQuery $theInsertCommand
		}
	}

	function fixForeignKeys($table, $otherTable, $column, $theToColumn)
	{
		if ($table -eq "mm_arecords" -and $otherTable -eq "mm_ipaddress")
		{
			## This may fix the problem outright (we do not prompt as these are superflous records)
			$theARecordFix = "delete from mm_arecords where recordid in (select recordid from (select recordid, count(ipaddressid) from mm_arecords group by recordid having count(ipaddressid) > 1)) and ipaddressid not in (select ipaddressid from mm_ipaddress) and recordid in (select notmissing from (select a.recordid notmissing, a.ipaddressid, i.ipaddressid from mm_arecords a, mm_ipaddress i where a.ipaddressid = i.ipaddressid));"
			InvokeNonQuery $theARecordFix
			
			## Check if this fixed it
			$theCommand = [string]::Format("select {0} from {1} where {0} not in (select {0} from {2});", $column, $table, $otherTable);
			$rows = InvokeAndReturnRows $theCommand
			if($rows.Count -ne $null){
				Write-Host "Will attempt to fix by adding entries to the parent table mm_ipaddress (if any are missing), and updating foreign key via the table mm_records" 
				$theAddressCommand = "select data from mm_records where recordtype in (1,28) and data not in (select dotted from mm_ipaddress)";
				$theMissingAddresses = InvokeAndReturnRows $theAddressCommand
				$theFixedAddressCount = 0
				if($theMissingAddresses.Count -ne $null){
					foreach ($row in $theMissingAddresses)
					{
						$dotted = $row.data
						$theIPAddress = [System.Net.IPAddress]::Parse($dotted);
						AddToIPAddress ($theIPAddress)
						$theFixedAddressCount += 1
					}
					
					$theMessage = [string]::Format("Added {0} record(s) in table mm_ipaddress.", $theFixedAddressCount.ToString())
					Write-Host $theMessage
				}
			}
			else
			{
				Write-Host "The table mm_arecords was automatically fixed by removing superflous records."
				return $true
			}

			if($theMissingAddresses.Length -ne $null -and $theMissingAddresses.Length -ne $theFixedAddressCount){	
				$theMessage = [string]::Format("There are still {0} record(s) in table mm_arecords failing foreign key constraints.", $theMissingAddresses.Length - $theFixedAddressCount)
				Write-Host $theMessage
			}
			else
			{
				$theARecordsUpdateCommand = "update mm_arecords set ipaddressid = (select addressid from (select a.recordid rid, i.ipaddressid addressid, a.ipaddressid missingid from mm_ipaddress i, mm_arecords a, mm_records r where a.recordid = r.recordid and i.dotted = r.data and not addressid = missingid and missingid not in (select ipaddressid from mm_ipaddress)) where rid = recordid and missingid = ipaddressid) where ipaddressid not in (select ipaddressid from mm_ipaddress);"
				InvokeNonQuery $theARecordsUpdateCommand
				return $true
			}
			trap
			{
				Write-Host "Error when trying to fix mm_ipaddress: $_"  @extraArgs
				return $false
			}
		}
		$extraArgs = @{}
	  	$extraArgs["foregroundColor"] = "Yellow"
		Write-Host ""
		Write-Host "Would you like to remove the the rows which fail foreign key constraints?"  @extraArgs
		Write-Host "It is recomended that you have a backup of the database before doing this operation, as this "  @extraArgs
		Write-Host "operation can not be undone!"  @extraArgs
		$resp = Read-Host "[Y]es or [N]o" 
		Write-Host ""
		if($resp -eq "y" -or $resp -eq "yes") {	
			$theCommand = [string]::Format("delete from {0} where {1} not in (select {3} from {2});", $table, $column, $otherTable, $theToColumn)
            loggit "fixForeignKey: $theCommand"
			InvokeNonQuery $theCommand
			return $true
		} else {
			Write-Host "The foreign key constraints have not been fixed. Database migration cannot proceede."
			return $false
		}
	}

	$redArgs = @{}
	$redArgs["foregroundColor"] = "Red"


	$tableRows = InvokeAndReturnRows "select name from sqlite_master where type = 'table' and name like 'mm_%'"
	$tables=@()
	foreach ($row in $tableRows){$tables += $row.name}
	$NFailedTebles = 0

	$tableCount = $tables.Count
	$checkedTables = 0;
	Write-Host ""
	Write-Host "Checking database $sourceDbFile. $tableCount tables."
	Write-Host ""

	if($tableCount -le 0) { 
		Write-Host ""
		Write-Host "Can not find tables in the target database $sourceDbFile!"
		Write-Host "Make sure that that the database file is correct and contains Men and Mice data."
		Write-Host ""
		exit 1
	}

	## Check the column sizes
	foreach($table in $tables) {
		write-progress -id 1 -activity "Checking tables for column sizes." -status "Checking $table ($checkedTables of $tableCount tables done)." -percentComplete ( $checkedTables/$tableCount *100)
		$pragmas = InvokeAndReturnRows "pragma table_info($table);"
		foreach($pragma in $pragmas) {
		
			if($pragma.type.ToLower().StartsWith("varchar(")) {
				$maxLength = $pragma.type.ToLower().Replace("varchar(","").Replace(")","")
				
				$column = $pragma.name
				foreach($specialCase in $specialCases)
				{
					if($table -eq $specialCase[0] -and $column -eq $specialCase[1] ){
						# Write-Host "changing length from $maxLength to "+ $specialCase[2]
						$maxLength = $specialCase[2]				
					}
				}
				write-progress -id 2 -parentId 1 -activity  "Checking column $column." -status " "
				$command = "select count(*) from $table where length( $column) > $maxLength "
				#Write-Host $command 
				$check = InvokeScalar $command
				if($check -eq 0 ){
					#Write-Host "Table $table OK."
				} else {				
					$nrLines = $check
				
					Write-Host "Error in database."  @redArgs
					Write-Host "Table $table has $nrLines to long values in column $column see folowing command : $command" @redArgs
	                try {
	                    $res = truncColumnWithCheck $table.ToString() $column.ToString() $maxLength
	                } catch {
	                    Write-Host "Unable to truncate the values in table $table." @redArgs
	                    $res = clearRowsWithCheck $table.ToString() $column.ToString() $maxLength
	                }
					
					if($res -eq 0) {
						$NFailedTebles++
					}
				}
			}
		}
		$checkedTables++
	}
	## Check the column sizes

	if($NFailedTebles -gt 0){
		Write-Host ""
		Write-Host "Check failed! there where $NFailedTebles table(s) with too large data fields." @redArgs
		Write-Host "Please fix before procceding with migration." @redArgs
		Exit 1
	} 

	Write-Host ""
	Write-Host "No tables with too large data fields."
	Write-Host ""

	if (-not $skipForeignKeyCheck){
		Write-Host "Checking integrity of constraints..."

		## Check the foreign key constraints
		$checkedTables = 0;
		foreach($table in $tables) {

			write-progress -id 1 -activity "Checking tables for constraints." -status "Checking $table ($checkedTables of $tableCount tables done)." -percentComplete ( $checkedTables/$tableCount *100)
			
			$pragmas = InvokeAndReturnRows "pragma foreign_key_list($table);"
			if ($pragmas -ne $null)
			{
				if ($pragmas.GetType().Name.Equals("String"))
				{
					$pragmas = @($pragmas)
				}
				foreach($pragma in $pragmas) {
					
					if(-not ([string]::IsNullOrEmpty($pragma.table)) -and -not ([string]::IsNullOrEmpty($pragma.from))) {
						$theOtherTable = $pragma.table
						$theColumn = $pragma.from
                        $theToColumn = $pragma.to
                        if(([string]::IsNullOrEmpty($pragma.to))) {$theToColumn = $theColumn}
						$theSpecialCase = [string]::Empty
						if ($table -eq "mm_access")
						{
							$theSpecialCase = " and not identityid = 4294967295 " 
						}
						$theCommand = [string]::Format("select count({0}) from {1} where {0} not in (select {4} from {2}) {3};", $theColumn, $table, $theOtherTable, $theSpecialCase, $theToColumn);
						write-progress -id 2 -parentId 1 -activity  "Checking foreign key constraint for tables $table, $theOtherTable on column $theColumn." -status $theCommand
					    #Write-Host $theCommand 
						$check = InvokeScalar $theCommand
						if($check -eq 0){
							#Write-Host "Table $table OK."
						} else {				
							$nrLines = $check
						
							Write-Host "Error in database."  @redArgs
							Write-Host "Table $table has $nrLines entries which do not fulfill constrain on foreign key $theColumn" @redArgs
                            Write-Host $theCommand
							$res = fixForeignKeys $table $theOtherTable $theColumn $theToColumn
							if($res -eq 0) {
								$NFailedTebles++
							}
						}
					}
				}
			}
			$checkedTables++
		}
	}

	if($NFailedTebles -gt 0){
		Write-Host ""
		Write-Host "Check failed! there where $NFailedTebles table(s) which failed foreign key constraints." @redArgs
		Write-Host "Please fix before procceding with migration." @redArgs
		Exit 1
	} 

	Write-Host ""
	Write-Host "Data check is complete. Database migration can procede."
	Write-Host ""
}


function DoInvoke($command)
{
	$cmdObj = $MSConn.CreateCommand()
	$cmdObj.CommandText = $command
	$cmdObj.CommandTimeout = 600
	$cmdObj.ExecuteNonQuery()
}


function getSQLiteDBVersion ($sourceDBPath){
	$cn = getSQLiteConnection
	$cm =  getSQLiteCommand "select value from mm_preferences where key = 'dbversion';" $cn
	$dbVersion = $cm.ExecuteScalar()

	$cm.CommandText = "select value from mm_preferences where key = 'dbextraversion';"
	$dbExtraVersion = $cm.ExecuteScalar()

	#$cn.close()

	return "$dbVersion.$dbExtraVersion"
}


function CheckDbVersions($sourceDbFile, $database )
{
	if ($ignoreDBVersion){
		return $true
	}

	$cmdObj = $MSConn.CreateCommand()
	$cmdObj.CommandText = "select value from $dbSchema.mm_preferences where `[key`] =`'dbversion`'"
	$dbVersion = $cmdObj.ExecuteScalar()

	$cmdObj = $MSConn.CreateCommand()
	$cmdObj.CommandText = "select value from $dbSchema.mm_preferences where `[key`] =`'dbextraversion`'"
	$dbExtraVersion = $cmdObj.ExecuteScalar()

	$destDbVersion = "$dbVersion.$dbExtraVersion"
	$sourceDbVersion = getSQLiteDBVersion $sourceDbFile


	loggit "The source database version is $sourceDbVersion and the destination database version is $destDbVersion"
	return ($destDbVersion -eq $sourceDbVersion)


	trap
	{
		loggit "Received an exception: $_.  Exiting." -logToConsole -asError
		exit 1
	}
}

function CheckDBSchema (){

	$cmdObj = $MSConn.CreateCommand()
	$cmdObj.CommandText = 'select schema_name();'
	$curSchema = $cmdObj.ExecuteScalar()

	if ($curSchema -ne $dbSchema){
		loggit "Default schema for user $username is [$curSchema] in database connection, but requested schema is [$dbSchema]." -logToConsole -asError
		return $false
	} else {
		return $true
	}
}


function createTempTable($table) {
	$command = ("create temporary table {0}_import as select * from {0};" -f $table)
	#Write-Host "Creating import temp table from $table $command"
	InvokeNonQuery $command
}

function dropTempTable($table) {
	$command = ("drop table temp.{0}_import;" -f $table)
	#Write-Host "Dropping temporary table $table_import" 
	InvokeNonQuery $command
}

$global:psqlcn = $null
function getPSSQLConnection($connstr) {
	if($global:psqlcn -eq $null){
		$conn=New-Object System.Data.Odbc.OdbcConnection
		#$connstr = "Driver={PostgreSQL Unicode(x64)};Server=localhost;Port=5432;Database=mmsuite;Uid=postgres;Pwd=abc123;"
		$conn.ConnectionString= $global:connString
		$conn.open()
		$global:psqlcn = $conn
	}
	return $global:psqlcn
}

function InvokePSQLQuery($query){
  $conn = getPSSQLConnection
  $cmd=new-object System.Data.Odbc.OdbcCommand($query, $conn)
  $cmd.CommandTimeout=120
  $ds=New-Object system.Data.DataSet
  $da=New-Object system.Data.odbc.odbcDataAdapter($cmd)
  [void]$da.fill($ds)
  return ($ds.Tables[0])
}

function InvokePSQLNonQuery($nonquery){
	$conn = getPSSQLConnection
	$cmd=new-object System.Data.Odbc.OdbcCommand($nonquery, $conn)
	$cmd.CommandTimeout=120
	$cmd.ExecuteNonQuery()
}

# returns the string to reset the squence for $table
function GetPSQLSeqReset($table) {
	$query = @"
	SELECT  'SELECT SETVAL(' ||quote_literal(S.relname)|| ', (SELECT GREATEST(MAX(' ||quote_ident(C.attname)|| '),1)  FROM ' ||quote_ident(T.relname)  || '));'
	FROM pg_class AS S, pg_depend AS D, pg_class AS T, pg_attribute AS C WHERE S.relkind = 'S'
AND S.oid = D.objid
AND D.refobjid = T.oid
AND D.refobjid = C.attrelid
AND D.refobjsubid = C.attnum
AND T.relname = '{0}'
ORDER BY S.relname;
"@ -f ($table)
	return (InvokePSQLQuery -query $query).ItemArray
}

$global:connString = ""
$global:psqlCString = ""
function createConnectionStrings(){
	if($tcpPort -eq "") {
		loggit ("Setting PSQL port to default 5432")
		$tcpPort = "5432"
	}
	$SecurePassword = Read-Host -Prompt "Enter password for $username on PostgreSQL on $ServerInstance" -asSecureString
	$basicStr = [System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($SecurePassword)
	$password = [System.Runtime.InteropServices.Marshal]::PtrToStringAuto($basicStr)
	[System.Runtime.InteropServices.Marshal]::FreeBSTR($basicStr)
	$global:connString = ("Driver={{PostgreSQL Unicode(x64)}};Server={0};Port={1};Database={2};Uid={3};Pwd={4}" -f $ServerInstance, $tcpPort, $database, $username, $password)
	$global:psqlCString = ("postgresql://{0}:{1}@{2}:{3}/{4}" -f $username, $password, $ServerInstance, $tcpPort, $database) 
	loggit ("Connecting to PSQL Server as {0}" -f ($username)) -logToConsole
}


# PSQL data import
function commitDataTablePSQL($dt, $tableName) {
	# we replace all value "-1" with the 4294967295 (= 0xffffffff)
	$outp = ($dt|ConvertTo-Csv -NoTypeInformation).replace('"-1"','"4294967295"')
	$outp|Set-Content -Path .\tables\$tableName 
	loggit("Exported {0} to into CSV file .\tables\{0}" -f $tableName)
#	$tableToReplaceMinusOne = @("mm_access", "mm_identities", "mm_healthnotifications")
#		$outp = ($dt|ConvertTo-Csv -NoTypeInformation).replace('"-1"','"4294967295"')
#		$outp|Set-Content -Path .\tables\$tableName 
#	} else {
#    	$dt |Export-Csv -Path .\tables\$tableName -NoTypeInformation
#	}
	# avoid issue with squences and triggers and foreign key constraints when importing data.
	InvokePSQLNonQuery -nonquery "SET session_replication_role = replica;"
	# columns to import are stored in the headers array
	$headers = @()
	# columns that will be forced to null are stored in $forcenull
	$forcenull = @()
	# if a table contains one of the forcenullfields it will att it to the force to null array
	$forcenullfields = @("lastmodifieddate", "groupid","roleid", "userid", "clientidentifierid", "dhcpgroupid", "serverobjectid", 
	"objectid", "objecttype", "identityid", "allow", "deny", "port", "processeddate", "preference")
	foreach ($col in $dt.Columns){
		$headers +=($col.ColumnName)
		# if we find one of the forcenullfields we force them to NULL to avoid bigint to "" errors
		# as empty string in the CSV is not recognized as NULL
		if($forcenullfields -contains $col.ColumnName ){
			$forcenull += $col.ColumnName
		}
	}
	$headerstr = $headers -join ","

	# for the psql CSV import we need a full path the exported CSV
	$fullpath = (Get-ChildItem -Path ".\tables\$tableName").FullName

	# depending if we have to force columns to null we format the copy command and we also set the session to replica.
	if($forcenull.length -gt 0){
		$insertstr = "set session_replication_role=replica;copy {0} ({1}) from '{2}' with (format CSV, Header, force_null({3}));" -f ($tableName, $headerstr, $fullpath, ($forcenull -join ","))
	} else {	
		$insertstr = "set session_replication_role=replica;copy {0} ({1}) from '{2}' with (format CSV, Header);" -f ($tableName, $headerstr, $fullpath)
	}
	# write-host $insertstr
    # the the client encoding to LATIN1 to allow psql the connection to the mmsuite database, which is LATIN1 with C or Posix.
	$env:PGCLIENTENCODING = "LATIN1"
	# execute the command in psql.exe The connection string is need as it does otherwise not accept a password.
	#$ret = & 'C:\Program Files\PostgreSQL\12\bin\psql.exe'  --command $insertstr postgresql://postgres:abc123@localhost:5432/mmsuite
	loggit ("import CSV file with psql {0}" -f $insertstr)
	$ret = & ($psqlCMDpath)  --command $insertstr $global:psqlCString
	# If we don't get a COPY back we stop and throw an exception.
	if($ret -eq $null -or (-not ($ret.contains("COPY")))) {
		throw "Stop: Error during CSV import of table " + $tableName
	}
}


# Microsoft data import
function commitDataTable($dt,$tableName){
	$totalRows = $dt.Rows.Count
	$bulkCopy = new-object Data.SqlClient.SqlBulkCopy($MSConn,([System.Data.SqlClient.SqlBulkCopyOptions]::KeepIdentity,[System.Data.SqlClient.SqlBulkCopyOptions]::TableLock),$null)
	$bulkCopy.DestinationTableName = $dbSchema+'.[' + $tableName + ']'
	$bulkCopy.BulkCopyTimeout = 3600
	$bulkCopy.NotifyAfter = 50000
	$bulkCopy.BatchSize = 50000
	
	$ev = Register-ObjectEvent -InputObject $bulkCopy -EventName 'SqlRowsCopied' -MessageData @{totalRows=$totalRows;tableName=$tableName} -Action { 
		$rowsCopied = $eventArgs.RowsCopied
		$totalRows = $event.MessageData.totalRows
		write-progress -id 3 -activity "Copying data to SQL Server" -status "Copied $rowsCopied/$totalRows rows to server." -PercentComplete (100*$rowsCopied/$totalRows) -parentId 2
	}


	#set the column mapping in case the ORDER is different between MSSQL and SQLite.
	#this does not handle cases where there is not the same amount of columns in SQL server and SQLite
	foreach ($col in $dt.Columns){
		$colName = $col.ColumnName
		$colMapping = New-Object System.Data.SqlClient.SqlBulkCopyColumnMapping($colName,$colName)
		$junk = $bulkCopy.ColumnMappings.Add( $colMapping )
	}

	write-progress -id 3 -activity "Copying data to SQL Server" -status "Starting to copy $totalRows rows to server" -parentId 2 -PercentComplete 0

	$tryCount=0
	while ($true){
		try{
			$tryCount++
			$bulkCopy.WriteToServer($dt)
			break
			write-progress -id 3 -activity "Done copying $tableName to SQL Server" -status "Done" -parentId 2 -PercentComplete 100
		} catch {
			if ($tableName -eq 'mm_zones' -and $_.Exception.Message.Contains('The given ColumnMapping does not match up with any column in the source or destination')){
				#attempt to add and drop column secure64 automatically, when there's a columnmapping mismatch. It is very likely because of MM-9660
				try {
					loggit ("Got column mismatch exception for mm_zones. Trying to apply fix.") -logToConsole
					DoInvoke ("Alter table $dbSchema.mm_zones add secure64 smallint;") > $null
					$bulkCopy.WriteToServer($dt);
					DoInvoke ("Alter table $dbSchema.mm_zones drop column secure64;") > $null
					loggit ("Column mismatch fix successfully applied") -logToConsole
					break
				} catch {
					loggit ('Got exception while trying to fix column mismatch exception: {0}.' -f $_.Exception.Message) -logToConsole -asError
					loggit "Aborting. Please contact M&M support." -asError -logToConsole
				}
			}
			if ($_.Exception.Message.Contains('Could not access destination table')){
				loggit ("Could not access destination table. Database is busy. Retrying in 2 seconds. Retry#{0}" -f $tryCount) -logToConsole -asError
			} else {
				loggit ("Encountered error: {0}. Retrying in 2 seconds. Retry#{1}" -f $_.Exception.Message,$tryCount) -logToConsole -asError
			}

			sleep -Seconds 2
		}
	}
	$bulkCopy.Close()
}


function DeleteAllFromDestinationTable ($database, $tableName)
{
	if ($tableName.ToLower() -eq "mm_identities") {
		return (DoInvoke "DELETE FROM $dbSchema.$tableName where identityid < 4294967295;")
	} else{
		try {
			return (DoInvoke "TRUNCATE table $dbSchema.$tableName;")
		} catch {
			return (DoInvoke "DELETE FROM $dbSchema.$tableName;")
		}
	}
}

# singleton for the SQLite connection
$global:sqlitecn = $null
function getSQLiteConnection() {
	if($global:sqlitecn -eq $null) {
		$global:sqlitecn = new-object System.Data.SQLite.SQLiteConnection
		$global:sqlitecn.ConnectionString = $sqliteConnStr
		$global:sqlitecn.open()
		write-host "Open Connection"
	}
	# write-host "Returning exsting connection"
	return $global:sqlitecn
}

function getSQLiteCommand($query,$connection){
	$cm =  New-Object System.Data.SQLite.SQLiteCommand
	$cm.CommandText = $query
	$cm.Connection = $connection
	# write-host $query
	return $cm
}



function getSQLiteTables ($sourceDBPath){
	$cn = getSQLiteConnection
	$cm = getSQLiteCommand "SELECT name FROM sqlite_master WHERE type='table' and name like 'mm_%' ORDER BY name;" $cn
	$da = new-object System.Data.SQLite.SQLiteDataAdapter($cm)
	[System.Data.DataTable]$dt = new-object System.Data.DataTable('tables')
	[void]$da.Fill($dt)
	#$cn.close()

	$tables = @()
	foreach ($row in $dt.Rows){
		$tables += $row.name
	}
	return $tables
}


Function Get-Bits(){
	try {
	    Switch ([System.IntPtr]::Size) {
        4 {
            Return 32
        }

        8 {
            Return 64
        }

        default {
	            Return 32
        }
    }
	} catch {
		return 32
	}
}



if (-not (Test-Path $sourceDbFile)){
	loggit "Could not find source database file" -logToConsole -asError
	exit 1
} else {
	$sourceDbFile = Convert-Path $sourceDbFile
}


$bits = Get-Bits
if ($bits -eq 64){
	$sqliteDLLPath = Convert-Path "./x64/System.Data.SQLite.DLL"
} elseif ($bits -eq 32){
	$sqliteDLLPath = Convert-Path "./x32/System.Data.SQLite.DLL"
} else {
	loggit "Unable to determine computer arcitecture (i.e., 32 or 64 bit). Exiting..." -logToConsole -asError
	exit 2
}


if (-not (Test-Path $sqliteDLLPath)){
	loggit "Could not find the System.Data.SQLite library dll at $sqliteDLLPath" -logToConsole -asError
	exit 3
}

if (-not (Get-Command -Name Add-Type -ErrorAction SilentlyContinue)) {
	$oldea = $ErrorActionPreference
	$ErrorActionPreference = 'Stop'
	[Reflection.Assembly]::LoadFrom($sqliteDLLPath)
	$ErrorActionPreference = $oldea
} else {
	Add-Type -Path $sqliteDLLPath
}


$oldea = $ErrorActionPreference
$ErrorActionPreference = 'Stop'

# the SQLite to MSSQL migration loop
if($DBType -eq "MSSQL") {

    #Create and open the Connection to SQL Server
    $MSConn = New-Object System.Data.SqlClient.SqlConnection;
    $proto = ''
    if ($forceTCP){ $proto = 'tcp:' }
    if ($forceNamedPipes){ $proto='np:' }
    if ($forceSharedMemory) {$proto='lpc:'}

    $tcpPortStr=''
    if ($tcpPort){ $tcpPortStr = ",$tcpPort" }

    if ($useWindowsAuthentication -or $username -eq $null ) { 
	    loggit ("Connecting to SQL Server as {0} with Integrated Security" -f (whoami)) -logToConsole
	    $connString = ("Server={0}{1}{2};Database={3};Integrated Security=True" -f $proto,$ServerInstance,$tcpPortStr,$database)
    } else {
	    $SecurePassword = Read-Host -Prompt "Enter password for $username on SQL Server" -asSecureString
	    $basicStr = [System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($SecurePassword)
        $password = [System.Runtime.InteropServices.Marshal]::PtrToStringAuto($basicStr)
        [System.Runtime.InteropServices.Marshal]::FreeBSTR($basicStr)
        $connString = ("Server={0}{1}{2};Database={3};Integrated Security=False;User Id={4};Password={5}" -f $proto,$ServerInstance,$tcpPortStr,$database,$username,$password)
        loggit ("Connecting to SQL Server as {0}" -f ($username)) -logToConsole
    }


    write-verbose $connString
    $MSConn.ConnectionString = $connString;
    $MSConn.Open();

    $sqliteConnStr = "Data Source=$sourceDBFile;"

    $cwd = pwd
    $myName = "ConvertDatabase3"
    $logfile = Join-Path $cwd "$myName.log"

    Write-Host "Log entries will be written out to the file $logfile"
    loggit 'Starting' 


    write-progress -id 1 -activity "Comparing database versions and schema." -status " "
    $versionCheck = CheckDbVersions $sourceDbFile $database

    if (-not $versionCheck)
    {
	    loggit "Versions are not identical! Please update the databases to current release and try data migration again, or use the ignoreDBVersion switch" -logToConsole -asError
	    Exit 4
    }

    $schemaCheck = checkDBSchema
    if (-not $schemaCheck){
	    $ans = Read-Host "Do you want to continue anyway with the data migration? `nData will be migrated into the $dbSchema database schema. [y/n]"
	    if ($ans.ToLower() -ne 'y'){
		    Exit 0
	    }
    }

    write-progress -id 1 -activity "Comparing database versions and schema." -status "Db versions Ok"




    Write-Host "Do you want to check the source database $sourceDbFile for issues that compromise compatibility with a SQL Server database?"
    Write-Host "(Note that if this check has been performed for this source database file once, it does not need to be done again.)"
    $ans = Read-Host "[Y]es or [N]o"
    if ($ans.ToLower() -eq 'y'){
	    loggit "Running CheckDatabaseForMigration"
	    CheckDatabaseForMigration 
    }



    $tables  = getSQLiteTables $sourceDbFile
    if (-not $skipPurgingTables){
	    Write-Host "This will purge all data from all tables in the destination database $database in SQL Server. Are you sure you want to continue?"
	    $ans = Read-Host "[Y]es or [N]o"
	    if ($ans.ToLower() -ne 'y'){
		    exit 0
	    }

	    $tablesDone = 0
	    $totalTables = $tables.Count
	    write-verbose 'Disabling constraints'
	    DoInvoke "exec sp_MSforeachtable `'ALTER TABLE ? NOCHECK CONSTRAINT ALL`';" > $null
	    foreach ($table in $tables){
		
		    write-progress -id 1 -activity "Purging database tables" -status "$table" -PercentComplete (($tablesDone/$totalTables)*100)
		
		    try {
			    if (isEmpty $table){
				    loggit "Empty table name?"
			    } else {
				    $numDeleted = DeleteAllFromDestinationTable $database $table
				    loggit ('{0} entries deleted from {1}' -f $numDeleted,$table)
			    }
		    } catch {
			    loggit ("Received an exception while deleting from table {0}: {1}.  Exiting." -f $table,$_.Exception.Message) -logToConsole -asError
			    exit 6
		    }

		    $tablesDone++
	    }
	    write-verbose "Re-enabling constraints"
	    DoInvoke "exec sp_MSforeachtable `'ALTER TABLE ? CHECK CONSTRAINT ALL`';" > $null

	    loggit "Database tables purged" -logToConsole
    } else {
	    loggit 'Database tables not purged. Assumed empty already.' -logToConsole
    }


    #restore the EA preference
    $ErrorActionPreference = $oldea

    loggit 'Starting data migration from SQLite to MS SQL Server' -logToConsole
    $tablesDone=0
    $totalTables = $tables.Count
    foreach ($table in $tables){
	    write-progress -id 1 -activity "Migrating tables to SQL Server" -status "Processing $table" -PercentComplete (($tablesDone/$totalTables)*100)

	    $theOffset = 0
	    $batchNr = 1

	    while ($true){

		    write-progress -id 2 -activity "Loading rows batch from SQLite" -status "Batch #$batchNr - $theOffset rows done" -parentId 1
		    #fetch data from sqlite (doing it here instead of in a special function, because the $dt variable is converted to Object[] when returned from a function)
		    $cn = getSQLiteConnection
		    $cm = getSQLiteCommand "select * from $table limit $batchSize offset $theOffset" $cn
		    $da = new-object System.Data.SQLite.SQLiteDataAdapter($cm)
		    [System.Data.DataTable]$dt = new-object System.Data.DataTable($table)
		    [void]$da.Fill($dt)
		    #$cn.close()

		    if ($dt.Rows.Count -gt 0){
			    write-progress -id 2 -activity "Copying $($dt.Rows.Count) rows to SQL Server" -status "Batch #$batchNr - $theOffset rows done" -parentId 1
			    commitDataTable $dt $table
			    $theOffset += $batchSize
			    $batchNr++
			    $dt.Clear()
			    [System.GC]::Collect()
		    } else {
			    break
		    }
	    }

	    $tablesDone++
	    loggit "Successfully imported table $table ($tablesDone/$totalTables)" -logToConsole
    }

    write-progress -id 1 -completed -Activity 'DONE'
    Write-progress -id 2 -Completed -Activity 'DONE'
    Write-progress -id 2 -Completed -Activity 'DONE'

    loggit "All Done!"
    $MSConn.Close()
# end of MSSQL 

# and here starts the PSQL migration loop
} elseif($DBType -eq "PSQL") {
	# Ask user for password and create the connection strings for direct psql.exe and ODBC connection.
	createConnectionStrings
    $sqliteConnStr = "Data Source=$sourceDBFile;"

    $cwd = pwd
    $myName = "ConvertDatabase3"
    $logfile = Join-Path $cwd "$myName.log"

    Write-Host "Log entries will be written out to the file $logfile"
    loggit 'Starting' 

    Write-Host "ToDO: Check DB versions"

    Write-Host "Do you want to check the source database $sourceDbFile for issues that compromise compatibility with a PSQL Server database?"
    Write-Host "(Note that if this check has been performed for this source database file once, it does not need to be done again.)"
    $ans = Read-Host "[Y]es or [N]o"
    if ($ans.ToLower() -eq 'y'){
	    loggit "Running CheckDatabaseForMigration"
	    CheckDatabaseForMigration 
    }

    $tables  = getSQLiteTables $sourceDbFile
    loggit 'Starting data migration from SQLite to PostgreSQL Server' -logToConsole
    $tablesDone=0
    $totalTables = $tables.Count
    foreach ($table in $tables){
	    write-progress -id 1 -activity "Migrating tables to PostgreSQL Server" -status "Processing $table" -PercentComplete (($tablesDone/$totalTables)*100)
		if(-not $skipPurgingTables) {
			loggit ("Deleteing content of table {0}" -f $table)
			$null = InvokePSQLNonQuery -nonquery ("delete from {0};" -f $table)
		}
		# get query to reset the sequence after the import of the table or null if no sequence is present
		$resetsequencequery  = GetPSQLSeqReset -table $table

	    $theOffset = 0
	    $batchNr = 1

		# to overcome the issue that default values might be set we create from the original table on the fly a temp table in sqlite <table name>_import 
        $ttable = $table + "_import"
        $null = createTempTable -table $table
		# migrate batches
	    while ($true){

		    write-progress -id 2 -activity "Loading rows batch from SQLite" -status "Batch #$batchNr - $theOffset rows done" -parentId 1
		    #fetch data from sqlite (doing it here instead of in a special function, because the $dt variable is converted to Object[] when returned from a function)
		    $cn = getSQLiteConnection
		    $cm = getSQLiteCommand "select * from temp.$ttable limit $batchSize offset $theOffset" $cn
		    $da = new-object System.Data.SQLite.SQLiteDataAdapter($cm)
		    [System.Data.DataTable]$dt = new-object System.Data.DataTable($ttable)
		    [void]$da.Fill($dt)

		    if ($dt.Rows.Count -gt 0){
			    write-progress -id 2 -activity "Copying $($dt.Rows.Count) rows to SQL Server" -status "Batch #$batchNr - $theOffset rows done" -parentId 1
                # write the batch CSV to PSQL 
                $null = commitDataTablePSQL -dt $dt -tableName $table
			    $theOffset += $batchSize
			    $batchNr++
			    $dt.Clear()
			    [System.GC]::Collect()
		    } else {
			    break
		    }
	    }
		# get rid of the temp _import table.
        $null = dropTempTable $table

		# if we have a squence on the PSQL table we fix it to max(id), i.e. the next will be max(id)+1
		if($resetsequencequery.length -ne 0){
			loggit ("reset squence: {0}" -f $resetsequencequery)
			$null = InvokePSQLNonQuery -nonquery $resetsequencequery
		}
	    $tablesDone++
	    loggit "Successfully imported table $table ($tablesDone/$totalTables)" -logToConsole
    }
	# close the DB connections to SQLite and PSQL
	$cn = getSQLiteConnection
	[void]$cn.close()
	$psqlcn = getPSSQLConnection
	[void]$psqlcn.close
	write-progress -id 1 -completed -Activity 'DONE'
	Write-progress -id 2 -Completed -Activity 'DONE'
    loggit "All Done!"

    
} else {
    throw ("Target DB type {0} is not supported. [supported targets are MSSQL and PSQL]" -f $DBType)
}

