$ErrorActionPreference = 'Stop'
cd $PSScriptRoot  > $null

function log($msg, $type = 'Information'){
	try {
		Write-EventLog -LogName Application -Source 'Men and Mice Updater' -EntryType $type -EventId 109 -Message $msg
	} catch { 
	}
	Add-Content -Value ('{0} [{1}] {2}' -f (get-date -Format 'yyyy-MM-dd HH:mm:ss'),$type,$msg) -Path 'C:\ProgramData\Men and Mice\Updater\mmUpdater-mmws.log'
}

function logInfo ($msg) {
	log -msg $msg -type 'Information' > $null
}
function logError ($msg) {
	log -msg $msg -type 'Error' > $null
}

function logErrorAndExit ($msg) {
	logError $msg
	
	Write-Output $msg
	exit 109
}

try {
	$newVersion = ./mmWS.exe -vv
} catch {
}
logInfo ('Updating Web Services and Web App to version {0}' -f $newVersion) 


# If there exist a new version of rest-doc in this package then we will 
# copy it to the correct place
if (Test-Path "rest-doc") {
	$mmwsDocPath = "C:\ProgramData\Men and Mice\Web Services\doc"
	$mmwsDocBakPath = $mmwsDocPath + ".bak"

	logInfo 'Updating REST documentation for web services' 
	if (Test-Path $mmwsDocPath) {
		# remove old backup if it exist and then rename current to backup
		try {
			Remove-Item -Force -Recurse $mmwsDocBakPath
		} catch {
		}
		try {
			Move-Item -Force -Path $mmwsDocPath -Destination $mmwsDocBakPath
		} catch {
			logInfo 'Problems moving the current documentation directory' 
		}
	}

	$wasSuccess = $false
	if (-Not (Test-Path $mmwsDocPath)) {
		try {
			Copy-Item -Force -Recurse -Path "rest-doc" -Destination $mmwsDocPath 
			$wasSuccess = $true
			logInfo 'REST documentation directory updated' 
		} catch {
			logInfo 'Problems copying the new REST documentation directory' 
		}
	}

	# if success remove backup dir else
	# we will try to revert
	if (Test-Path $mmwsDocBakPath) {
		try {
			if ($wasSuccess) {
				Remove-Item -Force -Recurse $mmwsDocBakPath	
			} else {
				Move-Item -Force -Path $mmwsDocBakPath -Destination $mmwsDocPath
			}
		} catch {
		}
	}
}

# Update scripts in extensions folder
if (Test-Path "extensions") {
	logInfo 'Updating scripts in the extensions folder for Central'
	Copy-Item -Force -Recurse -Path "extensions" -Destination "C:\ProgramData\Men and Mice\Central\extensions"
}

# Create scripts folder and copy README
if (Test-Path "scripts") {
	logInfo 'Updating scripts folder for Central'
	Copy-Item -Force -Recurse -Path "scripts" -Destination "C:\ProgramData\Men and Mice\Central\scripts"
}

# --mmWS.exe--
$mmwsName = "Men and Mice Web Services"
$mmwsService = Get-WmiObject win32_service | ? {$_.Name -eq $mmwsName} | Select-Object State, PathName
if (!$mmwsService){
	logErrorAndExit "Unable to find `"$mmwsName`" service"
}

$mmwsPath = $mmwsService.PathName.Trim('"')
$mmwsBak  = $mmwsPath + '.bak'

try {	
	logInfo ('Stopping Web Services')
	Stop-Service -Force -Name $mmwsName > $null
	logInfo ('Moving {0} to {1}' -f $mmwsPath, $mmwsBak)
	try {
		mv  $mmwsPath $mmwsBak -Force > $null
	} catch {
		# bak file is likely locked, try moving to mmWS.exe.<timestamp>.bak
		logError $_.Exception.Message
		$currentTimestamp = ((get-date -UFormat %s) -split '[.,]')[0]
		$mmwsBak = "{0}.{1}.bak" -f $mmwsPath, $currentTimestamp
		logInfo ('Moving mmWS.exe to bak file failed. Trying to move {0} to {1} instead' -f $mmwsPath, $mmwsBak)
		mv  $mmwsPath $mmwsBak -Force > $null
	}
	logInfo ('Moving new mmWS.exe to {0}' -f $mmwsPath)
	cp ./mmWS.exe $mmwsPath -Force > $null
	logInfo ('Starting Web Services')
	Start-Service $mmwsName > $null
} catch {
	logError $_.Exception.Message
	logError ("updated mmWS.exe could not be moved into place and/or started. Restarting service.")
	if (!(Test-Path $mmwsPath)){
		logInfo "Attempting to rollback updated mmWS before service restart"
		#we need to move the mmWS.exe.bak back to be able to start the service (we managed to move mmws to .bak but not copy the new one over)
		mv  $mmwsBak $mmwsPath -Force -ErrorAction Ignore > $null
	}
	Start-Service $mmwsName > $null
	logErrorAndExit "Update of mmWS and Web App aborted"
}


# --Web App-- Let's replace the directory
$webPath = $null
try {
	Import-Module WebAdministration # to support Win2008R2. Not needed on 2012/2012R2/2016, but doesn't hurt either
	$webPath = (Get-WebApplication -Name 'ui').physicalPath
} catch {}

if (!$webPath) {
	logErrorAndExit 'IIS not installed or Web App not found.'
}

try {
	logInfo ('Copying web.config from {0}' -f $webPath)
	cp (join-path $webPath 'web.config') (Join-Path './webapp' 'web.config') > $null
	$bakWebPath = $webPath+'.bak'

	logInfo ('Stopping IIS while files are copied')
	Stop-Service W3SVC -Force > $null
	#fetch the web.config from the M&M install dir and use that
	logInfo ('Moving {0} to {1}' -f $webPath,$bakWebPath)
	mv $webPath $bakWebPath -Force > $null
	logInfo ('Copying new Web App to {0}' -f $webPath)
	cp ./webapp $webPath -Recurse > $null
	logInfo ('Starting IIS')
	Start-Service W3SVC > $null

	logInfo ('Removing backup: {0}' -f $bakWebPath)
	rm -Recurse $bakWebPath > $null

	logInfo ('Successfully updated Web Services and Web App to version {0}' -f $newVersion) 
} catch {
	if (!(Test-Path $webPath)){
		#we need to restore the backup webapp dir. The copy of the new webapp seems to have failed
		mv $bakWebPath $webPath -Force > $null
	}
	Start-Service W3SVC > $null
	logErrorAndExit $_.Exception.Message
}
